/*
 * 'Auth' widget designed for the Linux-PAM project. Intended to do 
 * similar things to the 'Login' widget with xdm.
 *
 * Copyright Ben Buxton (bb@zip.com.au) 02/Feb/97
 * Updated: 03/28/1997 by Danny Sung <dannys@poboxes.com>
 * Cleaned up Andrew G. Morgan <morgan@parc.power.net> 1997/5/16
 *
 * See the file 'License' for License information 
 */

#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include "AuthP.h"

#define GRAB_KEYBOARD

#ifdef DEBUG
#include <stdio.h>
#define D(x) { \
    printf(__FILE__ " (" __FUNCTION__ ") %d: ", __LINE__); \
    printf x ; \
    printf("\n"); \
}
#else
#define D(x)
#endif

#ifndef min
#define min(a,b)  ( (a)<(b) ? (a):(b) )
#endif

/* Resource information */

static XtResource resources[] = {
#define offset(field) XtOffsetOf(AuthRec, field)
    { XtNbackground, XtCBackground, XtRPixel, sizeof(Pixel), 
    	offset(core.background_pixel), XtRString, "grey66" },
    { XtNborderWidth, XtCBorderWidth, XtRDimension, sizeof(Dimension), 
    	offset(core.border_width), XtRImmediate, (XtPointer) 0},
    { XtNlabelFore, XtCForeground, XtRPixel, sizeof(Pixel), 
    	offset(auth.l_foreground), XtRString, XtDefaultForeground },
    { XtNmessageFore, XtCForeground, XtRPixel, sizeof(Pixel), 
    	offset(auth.m_foreground), XtRString, XtDefaultForeground },
    { XtNmessageBack, XtCBackground, XtRPixel, sizeof(Pixel), 
    	offset(auth.m_background), XtRString, "grey80" },
    { XtNpromptFore, XtCForeground, XtRPixel, sizeof(Pixel), 
    	offset(auth.p_foreground), XtRString, XtDefaultForeground },
    { XtNentryFore, XtCForeground, XtRPixel, sizeof(Pixel), 
    	offset(auth.e_foreground), XtRString, XtDefaultForeground },
    { XtNshadowWidth, XtCShadowWidth, XtRDimension, sizeof(Dimension), 
    	offset(auth.shadow_width), XtRImmediate, (XtPointer) 2 },
    { XtNlabelFont, XtCFont, XtRFontStruct, sizeof(XFontStruct *), 
    	offset(auth.label_font), XtRString, 
    	"-adobe-new century schoolbook-bold-i-*-*-18-*-*-*-*-*-*-*"},
    { XtNpromptFont, XtCFont, XtRFontStruct, sizeof(XFontStruct *), 
    	offset(auth.prompt_font), XtRString, 
    	 "-adobe-helvetica-medium-o-*-*-18-*-*-*-*-*-*-*"},
    { XtNentryFont, XtCFont, XtRFontStruct, sizeof(XFontStruct *), 
    	offset(auth.entry_font), XtRString, 
    	"-adobe-helvetica-medium-r-*-*-18-*-*-*-*-*-*-*"},
    { XtNmessageFont, XtCFont, XtRFontStruct, sizeof(XFontStruct *), 
    	offset(auth.message_font), XtRString, 
    	"-adobe-new century schoolbook-medium-r-*-*-18-*-*-*-*-*-*-*" },
    { XtNbannerString, XtCLabel, XtRString, sizeof(String), 
    	offset(auth.banner_string), XtRString, "Welcome" },
    { XtNmessageString, XtCLabel, XtRString, sizeof(String), 
    	offset(auth.message_string), XtRString, (XtPointer) "" },
    { XtNentryMargin, XtCEntryMargin, XtRDimension, sizeof(Dimension), 
    	offset(auth.entrymargin), XtRImmediate, (XtPointer) 10 },
    { XtNokNotify, XtCCallback, XtRFunction, sizeof(XtPointer), 
    	offset(auth.ok_notify), XtRFunction, (XtPointer) 0 },
    { XtNabortNotify, XtCCallback, XtRFunction, sizeof(XtPointer), 
    	offset(auth.abor_notify), XtRFunction, (XtPointer) 0 },
    { XtNaddEchoString, XtCLabel, XtRString, sizeof(String), 
    	offset(auth.addEchoString), XtRString, NULL },
    { XtNaddLabelString, XtCLabel, XtRString, sizeof(String), 
    	offset(auth.addLabelString), XtRString, NULL },
    { XtNaddNoEchoString, XtCLabel, XtRString, sizeof(String), 
    	offset(auth.addNoEchoString), XtRString, NULL },
    { XtNclearPrompts, XtCBoolean, XtRBoolean, sizeof(Boolean),
	 	offset(auth.clearPrompts), XtRBoolean, (XtPointer) False },
    { XtNclearMessage, XtCBoolean, XtRBoolean, sizeof(Boolean),
	 	offset(auth.clearMessage), XtRBoolean, (XtPointer) False },
#undef offset
};

/* Declaration of methods */

static void Destroy(Widget w);
static void Initialize(Widget w, Widget new, ArgList args, Cardinal *num);
static void Realize(Widget w, Mask *valueMask, XSetWindowAttributes *attrib);
static void Redraw(Widget w, XEvent *event, Region region);
static void Resize(Widget w);
static Boolean SetValues(Widget cur, Widget req, Widget new, 
			 ArgList args, Cardinal *nargs);

/* private functions to AuthWidget */

static void CreateGCs(Widget w);
static void DrawCursor(Widget w, int pos, Boolean off); 
static void DrawEntry(Widget w);
static void DrawBanner(Widget w);
static void DrawMessage(Widget w);
static void DrawPrompt(Widget w, int pnum);
static void DrawShadows(Widget w, int x, int y, int width, int h, Boolean inv);
static void DrawScrews(Widget w);
static void InitBanner(Widget w);
static void InitMessage(Widget w);
static void NewPrompt(Widget w, String label, int type);

/* actions of AuthWidget */
static void Abort(Widget w, XEvent *event, String *parms, Cardinal *nparms);
static void InsertChar(Widget w, XEvent *event, String *parms,
		       Cardinal *nparms);
static void DeletePrev(Widget w, XEvent *event, String *parms,
		       Cardinal *nparms);
static void NextField(Widget w, XEvent *event, String *parms, Cardinal
		      *nparms);
static void PrevField(Widget w, XEvent *event, String *parms, Cardinal
		      *nparms);

/* a few important widget actions */
static XtActionsRec actions[] =
{
    {"insert-char", InsertChar},
    {"next-field", NextField},
    {"prev-field", PrevField},
    {"abort", Abort},
    {"delete-previous-char", DeletePrev},
};

static char defaultTranslations[] =
"Shift<Key>Tab:   prev-field() \n"
"<Key>Return:	   next-field() \n"
"<Key>F1:	      next-field(safe) \n"
"<Key>Tab:        next-field() \n"
"<Key>BackSpace:  delete-previous-char() \n"
"<Key>Delete:     delete-previous-char() \n"
"Ctrl<Key>H:      delete-previous-char() \n"
"<Key>:	         insert-char() \n"
;

AuthClassRec authClassRec = {
    { /* core fields */
        (WidgetClass) &widgetClassRec,   /* superclass               */
        "Auth",                          /* class_name               */
        sizeof(AuthRec),                 /* widget_size              */
        NULL,                            /* class_initialize         */
        NULL,                            /* class_part_initialize    */
        FALSE,                           /* class_inited             */
        Initialize,                      /* initialize               */
        NULL,                            /* initialize_hook          */
        Realize,                         /* realize                  */
        actions,                         /* actions                  */
        XtNumber(actions),               /* num_actions              */
        resources,                       /* resources                */
        XtNumber(resources),             /* num_resources            */
        NULLQUARK,                       /* xrm_class                */
        TRUE,                            /* compress_motion          */
        TRUE,                            /* compress_exposure        */
        TRUE,                            /* compress_enterleave      */
        FALSE,                           /* visible_interest         */
        Destroy,                         /* destroy                  */
        Resize,                          /* resize                   */
        Redraw,                          /* expose                   */
        SetValues,                       /* set_values               */
        NULL,                            /* set_values_hook          */
        XtInheritSetValuesAlmost,        /* set_values_almost        */
        NULL,                            /* get_values_hook          */
        NULL,                            /* accept_focus             */
        XtVersion,                       /* version                  */
        NULL,                            /* callback_private         */
        defaultTranslations,             /* tm_table                 */
        XtInheritQueryGeometry,          /* query_geometry           */
        XtInheritDisplayAccelerator,     /* display_accelerator      */
        NULL                             /* extension                */
    },
    { /* auth fields */
	0                                /* extension                */
    }
};

WidgetClass authWidgetClass = (WidgetClass)&authClassRec;
#define DIAMETER 7		/* Screw diameter */

#define ECHO 0
#define NOECHO 1
#define LABEL 2
#define ENTRYOFFSET 10	/* Offset between left edge of entry and text */
#define BUFSIZE 64		 /* Buffer size for username/password */

/*
 * Initialize some of the widget variables...the first function called
 * when the widget is created
 */

static void Initialize(Widget w, Widget new, ArgList args, Cardinal *num)
{
    AuthWidget aw = (AuthWidget) new;

    D(("called."));
    CreateGCs(new);
    aw->auth.numprompts = 0;

    /* Calculate label and message geometries */
    InitBanner(new);
    InitMessage(new);
    aw->auth.where = 0;
    aw->auth.cursorpos = 0;
    aw->auth.prompts = NULL;
    aw->auth.grabbed = FALSE;

    aw->auth.clearPrompts = 0;
    aw->auth.clearMessage = 0;
    D(("end."));
}

/* Free the resources we've used */
static void Destroy(Widget w)
{
    AuthWidget aw = (AuthWidget) w;
    Prompt *prompts;

    D(("called."));

    XtReleaseGC(w, aw->auth.topGC);
    XtReleaseGC(w, aw->auth.botGC);
    XtReleaseGC(w, aw->auth.normGC);
    XtReleaseGC(w, aw->auth.labelGC);
    XtReleaseGC(w, aw->auth.promptGC);
    XtReleaseGC(w, aw->auth.entryGC);
    XtReleaseGC(w, aw->auth.messageGC);
    XtReleaseGC(w, aw->auth.messagebGC);
    XFreePixmap(XtDisplay(w), aw->auth.screw);
	
    XUngrabKeyboard(XtDisplay(w), CurrentTime);

    while ((prompts=aw->auth.prompts) != NULL) {
	Prompt *p = (Prompt *)prompts->next;
	XtFree((void *)prompts);
	aw->auth.prompts = p;
    }
    aw->auth.numprompts=0;
    D(("end."));
}

/*
 * Called when the widget is realized - only called once per widget
 * just before it's actually displayed
 */

void Realize(Widget w, Mask *valueMask, XSetWindowAttributes *attrib)
{
    AuthWidget aw = (AuthWidget) w;
    Display *disp = XtDisplay((Widget)aw);
    Pixmap pix;
    Dimension x1 = DIAMETER/4;
    Dimension y1 = DIAMETER*0.75;
    Dimension x2 = y1;
    Dimension y2 = x1;
    Dimension tlx = 0;
    Dimension tly = 0;
    GC topgc = aw->auth.topGC;
    GC botgc = aw->auth.botGC;
    GC gc = aw->auth.normGC;
    GC pgc;
    XGCValues vals;
    XtGCMask mask;
    Prompt *prompt;

    D(("called."));

    /* Create the widget window */
    XtCreateWindow(w, InputOutput, (Visual *)CopyFromParent, *valueMask,
		   attrib);

    /* Create a pixmap containing image of the 'screw' */
    aw->auth.screw = XCreatePixmap(disp, XtWindow(w),
				   DIAMETER+1, DIAMETER+1,
				   DefaultDepthOfScreen(XtScreen(w)));
    pix = aw->auth.screw;
    mask = GCForeground;
    vals.foreground = aw->core.background_pixel;
    pgc = XtGetGC(w, mask, &vals);
    XFillRectangle(disp, pix, pgc, 0, 0, DIAMETER+1, DIAMETER+1);
    XDrawArc(disp, pix, botgc, tlx, tly, DIAMETER, DIAMETER, 2880, 11520);
    XDrawArc(disp, pix, topgc, tlx, tly, DIAMETER, DIAMETER, 14440, 11520);
    XDrawLine(disp, pix, gc, x1, y1+1, x2+1, y2);
    XtReleaseGC(w, pgc);
    /* Ignore initial labels */
    aw->auth.where = 0;
    for (prompt=aw->auth.prompts; prompt; prompt=(Prompt *)prompt->next) {
	if ( prompt->type == LABEL )
	    aw->auth.where++;
	else
	    break;
    }
    D(("end."));
}

static void Redraw(Widget w, XEvent *event, Region region)
{
    AuthWidget aw = (AuthWidget) w;
    Dimension width = aw->core.width;
    Dimension sw = aw->auth.shadow_width;
    Dimension h = aw->core.height;
    Dimension my = h-sw-aw->auth.mheight;
    Prompt *prompt = aw->auth.prompts;
    int num = 1;

    D(("called."));

    if (!XtIsRealized(w))
	return;

    /*
     * Grab the keyboard to prevent other clients from snooping
     * passwords, etc.  Not the best way but it works for most
     * stuff. The keyboard grab must be done here, as this is the only
     * time when we can be reasonably sure that the widget window is
     * viewable (necessary for a kbd grab)
     */

#if defined(GRAB_KEYBOARD)
    if (XtGrabKeyboard(w, FALSE, GrabModeAsync, GrabModeAsync, 
		       CurrentTime) != GrabSuccess) {
	if (!aw->auth.grabbed) {
	    XtWarning("XtGrabKeyboard failed\n");
	    aw->auth.grabbed = FALSE;
	}
    } else {
	aw->auth.grabbed = TRUE;
    }
#else
    aw->auth.grabbed = TRUE;
#endif

    /* Widget border */
    DrawShadows(w, 0, 0, width, h, FALSE);

    /* Banner and message borders */
    DrawShadows(w, sw, sw, width-sw, aw->auth.lheight, FALSE);
    DrawShadows(w, sw, my, width-sw, h-sw, TRUE);
    aw->auth.shadow_width/=2;

    /* Grooves above and below banner/message areas */
    DrawShadows(w, sw, my-sw+1, width-sw+1,
		my-sw+1, FALSE);
    DrawShadows(w, sw, aw->auth.lheight+1, width-sw+1,
		aw->auth.lheight+1, FALSE);
    aw->auth.shadow_width = sw;

    /* Banner screws */
    DrawScrews(w);

    /* Banner itself */
    DrawBanner(w);

    /* Message area */
    DrawMessage(w);

    /* Prompts */
    if (XRectInRegion(region, sw, aw->auth.lheight,
		      width, my-aw->auth.lheight)) {
	while (prompt != NULL) {
	    DrawPrompt(w, num);
	    prompt = (Prompt *)prompt->next;
	    num++;
	}
	DrawEntry(w);

	/* Cursor */
	DrawCursor(w, aw->auth.cursorpos+2, FALSE);
    }
    D(("end."));
}

/* Relayout the widget if it's resized */
static void Resize(Widget w)
{
    D(("called."));
    InitBanner(w);
    InitMessage(w);
    D(("end."));
}

/* Called when an app calls XtSetValues() on the widget */
static Boolean SetValues(Widget cur, Widget req, Widget new, ArgList args,
			 Cardinal *nargs)
{
    AuthWidget caw = (AuthWidget) cur;
    AuthWidget naw = (AuthWidget) new;
    Boolean redraw = FALSE;

    D(("called."));
    if (caw->auth.addEchoString != naw->auth.addEchoString ) {
	NewPrompt(new, naw->auth.addEchoString, ECHO);
	naw->auth.addEchoString = "";
	redraw = TRUE;
    }
    if (caw->auth.addLabelString != naw->auth.addLabelString ) {
	NewPrompt(new, naw->auth.addLabelString, LABEL);
	naw->auth.addLabelString = "";
	redraw = TRUE;
    }
    if (caw->auth.addNoEchoString != naw->auth.addNoEchoString ) {
	NewPrompt(new, naw->auth.addNoEchoString, NOECHO);
	naw->auth.addNoEchoString = "";
	redraw = TRUE;
    }

    if (caw->auth.label_font != naw->auth.label_font ||
	caw->auth.message_font != naw->auth.message_font) {
	XtReleaseGC(new, naw->auth.labelGC);
	XtReleaseGC(new, naw->auth.messageGC);
	InitBanner(new);
	InitMessage(new);
	redraw = TRUE;
    }

    if (caw->auth.message_string != naw->auth.message_string) {
	InitMessage(new);
	redraw = TRUE;
    }

    if (caw->auth.l_foreground != naw->auth.l_foreground
	|| caw->auth.m_foreground != naw->auth.m_foreground
	|| caw->auth.p_foreground != naw->auth.p_foreground
	|| caw->auth.e_foreground != naw->auth.e_foreground
	|| caw->auth.m_background != naw->auth.m_background) {
	XtReleaseGC(new, naw->auth.labelGC);
	XtReleaseGC(new, naw->auth.messageGC);
	XtReleaseGC(new, naw->auth.messagebGC);
	XtReleaseGC(new, naw->auth.normGC);
	redraw = TRUE;
    }

    if (caw->auth.shadow_width != naw->auth.shadow_width)
	redraw = TRUE;
    if (caw->auth.banner_string != naw->auth.banner_string) {
	InitBanner(new);
	redraw = TRUE;
    }

    if (caw->auth.clearPrompts != naw->auth.clearPrompts) {
	Prompt *p;

	while( (p=naw->auth.prompts) != NULL ) {
	    Prompt *tp = (Prompt *)p->next;
	    XtFree((void *)p);
	    naw->auth.prompts = tp;
	}
	naw->auth.numprompts = 0;
	naw->auth.where = 0;

	caw->auth.prompts = NULL;
	caw->auth.numprompts = 0;
	caw->auth.where = 0;

	naw->auth.clearPrompts = False;
	redraw = TRUE;
    }

    if (caw->auth.clearMessage != naw->auth.clearMessage) {
	naw->auth.message_string = "";
	naw->auth.clearMessage = False;
	InitMessage(new);
	redraw = TRUE;
    }

    if (redraw == TRUE)
	CreateGCs(new);
	
    D(("end."));
    return(redraw);
}


/* Create the various GCs for drawing attributes such as font, colour,etc */
static void CreateGCs(Widget w)
{
    AuthWidget aw = (AuthWidget) w;
    Display *disp = XtDisplay(w);
    Screen *screen = XtScreen(w);
    Colormap cmap = DefaultColormapOfScreen (screen);
    XColor topcolour;
    XColor botcolour;
    XColor colour;
    XGCValues GCV;
    XtGCMask	mask;

    D(("called."));

    /* Top shadow is twice the brightness of the background */
    topcolour.pixel = aw->core.background_pixel;
    if (topcolour.pixel == WhitePixelOfScreen (screen) ||
	topcolour.pixel == BlackPixelOfScreen (screen)) {
   	colour.red   = 32768;
   	colour.green = 32768;
   	colour.blue  = 32768;
    } else {
	XQueryColor (disp, cmap, &topcolour);
	colour.red   = topcolour.red/2;
	colour.green = topcolour.green/2;
	colour.blue  = topcolour.blue/2;
    }
    XAllocColor(disp, cmap, &colour);
    aw->auth.top_pixel = colour.pixel;
        
    /* Bottom shadow is half the brigtness */

    botcolour.pixel = aw->core.background_pixel;
    if (botcolour.pixel == WhitePixelOfScreen (screen) ||
	botcolour.pixel == BlackPixelOfScreen (screen)) {
   	colour.red   = 32768;
   	colour.green = 32768;
   	colour.blue  = 32768;
    } else {
   	XQueryColor (disp, cmap, &botcolour);
	colour.red   = min(65535, botcolour.red*2);
	colour.green = min(65535, botcolour.green*2);
	colour.blue  = min(65535, botcolour.blue*2);
    }
    XAllocColor(disp, cmap, &colour);
    aw->auth.bot_pixel = colour.pixel;

    mask = GCForeground;
    GCV.foreground = aw->auth.bot_pixel;
    aw->auth.botGC = XtGetGC(w, mask, &GCV);

    GCV.foreground = aw->auth.top_pixel;
    aw->auth.topGC = XtGetGC(w, mask, &GCV);

    /* Banner text GC */
    GCV.foreground = aw->auth.l_foreground;
    GCV.font = aw->auth.label_font->fid;
    mask |= GCFont;
    aw->auth.labelGC = XtGetGC(w, mask, &GCV);

    /* Prompt text GC */
    GCV.font = aw->auth.prompt_font->fid;
    aw->auth.normGC = XtGetGC(w, mask, &GCV);

    /* message area background GC */
    GCV.foreground = aw->auth.m_background;
    GCV.font = aw->auth.entry_font->fid;
    aw->auth.messagebGC = XtGetGC(w, mask, &GCV);

    /* Message area foreground */
    GCV.foreground = aw->auth.m_foreground;
    aw->auth.messageGC = XtGetGC(w, mask, &GCV);

    /* Prompt GC */
    GCV.font = aw->auth.prompt_font->fid;
    GCV.foreground = aw->auth.p_foreground;
    aw->auth.promptGC = XtGetGC(w, mask, &GCV);

    /* Text entry areas */
    GCV.font = aw->auth.entry_font->fid;
    GCV.foreground = aw->auth.e_foreground;
    aw->auth.entryGC = XtGetGC(w, mask, &GCV);

    D(("end."));
}

/* Draw a cursor (off -> 'Hide' the cursor) */
static void DrawCursor(Widget w, int pos, Boolean off)
{
    AuthWidget aw = (AuthWidget) w;
    GC gc;
    int which = aw->auth.where;
    int x, y, y1, y2;
    int sep; 		/* Distance between banner and message areas */
    XFontStruct *font = aw->auth.entry_font;
    int sw = aw->auth.shadow_width;
    int n = aw->auth.numprompts;
    int num;
    Prompt *prompt;

    D(("called."));
    if ((prompt = aw->auth.prompts)==NULL || n == 0 || which>=n)
	return;

    for (num=0; num<which && prompt->next != NULL; num++)
	prompt = (Prompt *)prompt->next;

    gc = off ? aw->auth.messagebGC : aw->auth.entryGC;
    x = pos + prompt->left;
    sep = aw->core.height - aw->auth.lheight - aw->auth.mheight;
    y = aw->auth.lheight + sep/(4*n) + (which)*sep/n +
	font->max_bounds.ascent + 1.5*sw;
    y1 = y - font->max_bounds.ascent + sw;
    y2 = y + font->max_bounds.descent - sw;
    XDrawLine(XtDisplay(w), XtWindow(w), gc, x, y1, x, y2);

    D(("end."));
}

/* Draw the entry text fully (after an expose) */
static void DrawEntry(Widget w)
{
    AuthWidget aw = (AuthWidget) w;
    int y, sep;
    GC gc = aw->auth.entryGC;
    int n = aw->auth.numprompts;
    Prompt *prompt;
    int num;
    XFontStruct *font = aw->auth.entry_font;

    D(("called."));
    sep = aw->core.height - aw->auth.lheight - aw->auth.mheight;
    prompt = aw->auth.prompts;
    num = 1;

    /* Do it for every prompt */
    while(num <= n) {
	y = aw->auth.lheight + sep/(4*n) + (num-1)*sep/n
	    +font->max_bounds.ascent + 1.5*aw->auth.shadow_width;
	if (prompt->type != NOECHO)
	    XDrawString(XtDisplay(w), XtWindow(w), gc, prompt->left,
			y, prompt->entrytext, strlen(prompt->entrytext));
	prompt = (Prompt *)prompt->next;
	num++;
    }

    D(("end."));
}

/* draw the banner text */
static void DrawBanner(Widget w)
{
    AuthWidget aw = (AuthWidget) w;
    String string = aw->auth.banner_string;

    D(("called."));
    XDrawString(XtDisplay(w), XtWindow(w), aw->auth.labelGC, 
		aw->auth.label_x, aw->auth.label_y,
		string, strlen(string));
    D(("end."));
}

/* Draw the message text/background */
static void DrawMessage(Widget w)
{
    AuthWidget aw = (AuthWidget) w;
    String string = aw->auth.message_string;
    int sw = aw->auth.shadow_width;
    int x = 2*sw;
    int y = aw->core.y + aw->core.height - aw->auth.mheight;
    unsigned int wd = aw->core.width - 4*sw;
    unsigned int h = aw->auth.mheight - 2*sw;

    D(("called."));
    XFillRectangle(XtDisplay(w), XtWindow(w), aw->auth.messagebGC,
		   x, y, wd, h);
    XDrawString(XtDisplay(w), XtWindow(w), aw->auth.messageGC, 
		aw->auth.message_x, aw->auth.message_y,
		string, strlen(string));
    D(("end."));
}

/* Draw the echo/no-echo prompt, carefully laid out (pnum = which prompt) */

static void DrawPrompt(Widget w, int pnum)
{
    AuthWidget aw = (AuthWidget) w;
    Display *disp = XtDisplay(w);
    Window win = XtWindow(w);
    GC pgc = aw->auth.promptGC;
    GC bgc = aw->auth.messagebGC;
    XFontStruct *font;
    int sw = aw->auth.shadow_width;
    int x, y, sep, ex, hsw, num;
    int n = aw->auth.numprompts;
    Dimension pwidth, pheight, ewidth=0, eheight;
    Prompt *prompt;
    String  label;
    Boolean entry;

    D(("called."));
    prompt = aw->auth.prompts;
    for (num=1; num<pnum; num++)
	prompt = (Prompt *)prompt->next;

    label = prompt->labeltext;
    font = aw->auth.prompt_font;
    if (prompt->type == ECHO || prompt->type == NOECHO)
	entry = TRUE;
    else
	entry = FALSE;
    x = aw->core.width/8;
    sep = aw->core.height - aw->auth.lheight - aw->auth.mheight;
    y = aw->auth.lheight + sep/(4*n) +  (pnum-1)*sep/n;
    pwidth = XTextWidth(font, label, strlen(label));

    /* Prompt */
    if (entry) {
	pheight = font->max_bounds.ascent + font->max_bounds.descent;
	XDrawString(disp, win, pgc, x, y + font->max_bounds.ascent + 1.5*sw,
		    label, strlen(label));
	ex = x + pwidth + aw->auth.entrymargin;
    } else {
	ewidth = pwidth + aw->auth.entrymargin;
	ex = aw->core.width/2 - ewidth/2;
	pwidth = 0;
    }

    /* Entry box */
    font = aw->auth.entry_font;
    if (entry)
	ewidth = 7*aw->core.width/8 - x - pwidth;
    eheight = (font->max_bounds.ascent + font->max_bounds.descent) + 3*sw;
    if (entry) {
	XFillRectangle(disp, win, bgc, ex, y, ewidth, eheight);
	DrawShadows(w, ex, y, ex+ewidth, y+eheight, TRUE);
    }
    prompt->left = ex + 5;
    prompt->right = prompt->left + ewidth - 10;

    hsw = sw/2;

    /* Half shadow width for outer - for a raised appearance */
    aw->auth.shadow_width /= 2;
    DrawShadows(w, ex-hsw, y-hsw, ex+ewidth+hsw, y+eheight+hsw, 1-entry);
    if (!entry)
	DrawShadows(w, ex, y, ex+ewidth, y+eheight, entry);
    aw->auth.shadow_width = sw;
    D(("end."));
}

/* Shadow drawing routine. Kindly borrowed from Xaw3d :) */
static void DrawShadows(Widget w, int x, int y,	int width, int h, Boolean inv)
{
    AuthWidget aw = (AuthWidget) w;

    D(("called."));
    if (aw->auth.shadow_width  > 0) {
	XPoint  pt[6];
	Dimension s   = aw->auth.shadow_width;
	Dimension ytl = y + s;
	Dimension xtl = x + s;
	Dimension ybr = h - s;
	Dimension xbr = width -s;
	GC		topgc;
	GC		botgc;
	Dimension       xms = xtl - s;
	Dimension       yms = ytl - s;
	Dimension       xps = xbr + s;
	Dimension       yps = ybr + s;
	Display         *dpy = XtDisplay (w);
	Window          win = XtWindow (w);

	if(inv) {
	    topgc = aw->auth.topGC;
	    botgc = aw->auth.botGC;
	} else {
	    topgc = aw->auth.botGC;
	    botgc = aw->auth.topGC;
	}

	pt[0].x = xms;  pt[0].y = yps;
	pt[1].x = xms;  pt[1].y = yms;
	pt[2].x = xps;  pt[2].y = yms;
	pt[3].x = xbr;  pt[3].y = ytl;
	pt[4].x = xtl;  pt[4].y = ytl;
	pt[5].x = xtl;  pt[5].y = ybr;
	XFillPolygon (dpy, win, topgc, pt, 6,Complex,CoordModeOrigin);
	pt[1].x = xps;  pt[1].y = yps;
	pt[4].x = xbr;  pt[4].y = ybr;
	XFillPolygon (dpy, win, botgc, pt,6, Complex,CoordModeOrigin);
    }
    D(("end."));
}

/* Copy the screw pixmap to appropriate areas on the window */
static void DrawScrews(Widget w)
{
    AuthWidget aw = (AuthWidget) w;
    GC gc = aw->auth.topGC;
    Display *disp = XtDisplay(w);
    Window pix = aw->auth.screw;
    Window win = XtWindow(w);
    Dimension sw = aw->auth.shadow_width*2;
    Dimension tlx = 1.5*sw;
    Dimension trx = aw->core.width - tlx - DIAMETER;
    Dimension bly = aw->auth.lheight - sw - DIAMETER;
    Dimension brx = aw->core.width - tlx - DIAMETER;

    D(("called."));
    XCopyArea(disp, pix, win, gc, 0, 0, DIAMETER+1, DIAMETER+1, tlx, tlx);
    XCopyArea(disp, pix, win, gc, 0, 0, DIAMETER+1, DIAMETER+1, trx, tlx);
    XCopyArea(disp, pix, win, gc, 0, 0, DIAMETER+1, DIAMETER+1, tlx, bly);
    XCopyArea(disp, pix, win, gc, 0, 0, DIAMETER+1, DIAMETER+1, brx, bly);
    D(("end."));
}

/* Initialize banner variables, mainly dealing with text positions */
void InitBanner(Widget w)
{
    AuthWidget aw = (AuthWidget) w;
    String lstring = aw->auth.banner_string;
    int label_l, label_h;
    Dimension sw = aw->auth.shadow_width;

    D(("called."));
    label_l = XTextWidth(aw->auth.label_font, lstring, strlen(lstring));
    label_h = aw->auth.label_font->max_bounds.ascent + 
	aw->auth.label_font->max_bounds.descent + 2*sw;
    aw->auth.label_x = (aw->core.width/2) - label_l/2 + 2*sw;
    aw->auth.lheight = 1.5*label_h;
    aw->auth.label_y = 2*aw->auth.lheight/3;
    D(("end."));
}

/* Do the same for the message area */
void InitMessage(Widget w)
{
    AuthWidget aw = (AuthWidget) w;
    String lstring = aw->auth.message_string;
    int message_l, message_h;
    Dimension sw = aw->auth.shadow_width;

    D(("called."));
    if (aw->auth.message_font == NULL)
	return;
    message_l = XTextWidth(aw->auth.message_font, lstring, strlen(lstring));
    message_h = aw->auth.message_font->max_bounds.ascent + 
	aw->auth.message_font->max_bounds.descent + 2*sw;
    aw->auth.message_x = aw->core.x + (aw->core.width/2) - message_l/2;
	
    aw->auth.mheight = 1.5*message_h;
    aw->auth.message_y = aw->core.y + aw->core.height - aw->auth.mheight/3;
    D(("end."));
}

/*
 * Allocate and initialize memory for a new 'prompt' args are widget,
 * label to display (or prompt to display) and type of prompt (ECHO,
 * NOECHO or LABEL)
 *
 * ( type = {echo, noecho, error, message} )
 */

static void NewPrompt(Widget w,	String label, int type)
{
    AuthWidget aw = (AuthWidget) w;
    Prompt *prompt;

    D(("called."));
    prompt = aw->auth.prompts;

    /* Linked list to hold prompt info */
    if (prompt == NULL) {
	/* Prompt list is empty - start it */
	aw->auth.prompts = (Prompt *)XtMalloc(sizeof(Prompt));
	prompt = aw->auth.prompts;

    } else {
	while(prompt->next != NULL) /* Jump to the end of the list */
	    prompt = (Prompt *)prompt->next;
	prompt->next = (struct Prompt *)XtMalloc(sizeof(Prompt));
	prompt = (Prompt *)prompt->next;
    }

    aw->auth.numprompts++;
    prompt->labeltext = XtMalloc(BUFSIZE);
    memset(prompt->labeltext, 0, BUFSIZE);
    strncpy(prompt->labeltext, label, strlen(label));
    prompt->entrytext = XtMalloc(BUFSIZE);
    memset(prompt->entrytext, 0, BUFSIZE);

    if (type == LABEL) {
	strncpy(prompt->labeltext, label, strlen(label));
	XtFree(prompt->entrytext);
	prompt->entrytext = prompt->labeltext;
    }
    prompt->type = type;
    prompt->next = NULL;

    /* Recalculate cursor position each time we add a new label */
    aw->auth.where = 0;
    for (prompt=aw->auth.prompts; prompt; prompt=(Prompt *)prompt->next) {
	if (prompt->type == LABEL)
	    aw->auth.where++;
	else
	    break;
    }
    D(("end."));
}


/* Calls the abort callback */
static void Abort(Widget w, XEvent *event, String *parms, Cardinal *nparms)
{
    AuthWidget aw = (AuthWidget) w;

    D(("called."));
    aw->auth.cursorpos = 0;
    if (aw->auth.abor_notify != NULL)
	(*aw->auth.abor_notify) (w);
    D(("end."));
}

/*
 * Widget action routine that inserts a new character into the line,
 * draws if necessary
 */
static void InsertChar(Widget w, XEvent *event, String *parms,
		       Cardinal *nparms)
{
    AuthWidget aw = (AuthWidget) w;
    Display *disp = XtDisplay(w);
    Window win = XtWindow(w);
    int len, width;
    char buf[16]; /* Buffer for character */
    GC gc;
    int sep, x, y;
    XFontStruct *font = aw->auth.entry_font;
    int echo, num;
    Prompt *prompt;
    int n = aw->auth.numprompts;
    int which = aw->auth.where;

    D(("called."));
    if ( n == 0 || which >= n)
	return;

    sep = aw->core.height - aw->auth.lheight - aw->auth.mheight;

    /* XLookupString doesn't return len bigger than sizeof(buf) */

    len = XLookupString(&event->xkey, buf, sizeof(buf), 0, 0);
    buf[len] = '\0';
    width = XTextWidth(font, buf, len);
    y = aw->auth.lheight + sep/(4*n) + which*sep/n
	+font->max_bounds.ascent + 1.5*aw->auth.shadow_width;

    prompt = aw->auth.prompts;
    num = 1;
    while (num <= which) {
	prompt = (Prompt *)prompt->next;
	num++;
    }
    x = aw->auth.cursorpos + prompt->left;

    if (prompt->type == ECHO) {	            /* Position for username box */
	echo = TRUE;

	/* Prevent buffer overflows */
	if (strlen(prompt->entrytext) + len >= BUFSIZE-1)
	    return;

	/* Username can't be longer than entry box */
	if (XTextWidth(font, prompt->entrytext, strlen(prompt->entrytext))+
	    prompt->left > prompt->right - ENTRYOFFSET) {
	    XBell(disp, 50);
	    return;
	}
	strncat(prompt->entrytext, buf, len);
    } else {	                                /* Position for password */

	/* Make sure password isnt longer than buffer size */
	if(strlen(prompt->entrytext) + len >= BUFSIZE-1)
	    return;
	strncat(prompt->entrytext, buf, len);
	echo = FALSE; /* Don't echo password */
    }

    if (echo) {                /* Echo the character and move the cursor */
	gc = aw->auth.entryGC;
	DrawCursor(w, x-prompt->left+2, TRUE);
	XDrawString(disp, win, gc, x, y, buf, strlen(buf));
	aw->auth.cursorpos += width;
	x = aw->auth.cursorpos;
	DrawCursor(w, x+2, FALSE);
    }

    D(("end."));
}

/* Delete the previous character from screen and buffer */
static void DeletePrev(Widget w, XEvent *event, String *parms,
		       Cardinal *nparms)
{
    AuthWidget aw = (AuthWidget) w;
    int width, sep, y;
    XFontStruct *font = aw->auth.entry_font;
    int len;
    GC gc = aw->auth.messagebGC;
    int n = aw->auth.numprompts;
    int which = aw->auth.where;
    int num;
    Prompt *prompt;

    D(("called."));
    if ( n == 0 || which >= n )
	return;

    prompt = aw->auth.prompts;
    for(num=0; num<which; num++)
	prompt = (Prompt *)prompt->next;

    len = strlen(prompt->entrytext);
    if (len < 1)
	return; /* return if there are no chars to delete */

    sep = aw->core.height - aw->auth.lheight - aw->auth.mheight;
    y = aw->auth.lheight + sep/(4*n) +  which*sep/n + 
	font->max_bounds.ascent + 1.5*aw->auth.shadow_width;
    width = XTextWidth(font, &prompt->entrytext[len-1], 1);
    DrawCursor(w, aw->auth.cursorpos+2, TRUE);

    if ( prompt->type != NOECHO ) {
	/* Cursor position back one */
	aw->auth.cursorpos -= width;

	/* Draw the last char in the BG colour */
	XDrawString(XtDisplay(w), XtWindow(w), gc,
		    aw->auth.cursorpos + prompt->left,
		    y, &prompt->entrytext[len-1], 1);
    }

    /* Make the last (non-NULL) character NULL to delete it */
    prompt->entrytext[len-1] = '\0';
    DrawCursor(w, aw->auth.cursorpos+2, FALSE);

    D(("end."));
}


/*
 * Moves the cursor to the next field, sets variables for new prompt
 * calls  the ok callback if we're on the password prompt
 */

static void NextField(Widget w, XEvent *event, String *parms, Cardinal *nparms)
{
    AuthWidget aw = (AuthWidget) w;
    Prompt *prompt=aw->auth.prompts;
    int num;
    Boolean end = TRUE;
    XFontStruct *font;
    char *str;

    D(("called."));
    DrawCursor(w, aw->auth.cursorpos+2, TRUE);

    /* Determine if we're at the last prompt - if so call the callbacks */
    for (num=0; num < aw->auth.where; num++)
	prompt = (Prompt *)prompt->next;

    while ( aw->auth.where < aw->auth.numprompts-1 ) {
	aw->auth.where++;
	prompt = (Prompt *)prompt->next;
	if (prompt->type == ECHO || prompt->type == NOECHO) {
	    end = FALSE;
	    break;
	}
    }

    if (end) {
	aw->auth.where = 0;

	for (prompt=aw->auth.prompts; prompt; prompt=(Prompt *)prompt->next) {
	    if ( prompt->type == LABEL )
		aw->auth.where++;
	    else
		break;
	}

	/*	Call the 'ok' callback function */
	if (*nparms > 0 && aw->auth.ok_notify != NULL) {
	    if (strncmp("safe", parms[0], 4))
		(*aw->auth.ok_notify) (w, aw->auth.prompts, LOGIN_SAFE);
	    else
		(*aw->auth.ok_notify) (w, aw->auth.prompts, LOGIN_OK);
	} else {
	    if (aw->auth.ok_notify != NULL)
		(*aw->auth.ok_notify) (w, aw->auth.prompts, LOGIN_OK);
	}
	num = 1;
	prompt = aw->auth.prompts;
	while (num <= aw->auth.numprompts) {
	    if (prompt->type != LABEL)
		memset(prompt->entrytext, 0, BUFSIZE);
	    DrawPrompt(w, num);
	    prompt = (Prompt *)prompt->next;
	    num++;
	}
    }

    if ( prompt ) {
	font = aw->auth.entry_font;
	str = prompt->entrytext;
	if( prompt->type != NOECHO )
	    aw->auth.cursorpos = XTextWidth(font, str, strlen(str));
	else
	    aw->auth.cursorpos = 0;
    }

    DrawCursor(w, aw->auth.cursorpos+2, FALSE);

    D(("end."));
}

/* Moves cursor to previous field */
static void PrevField(Widget w, XEvent *event, String *parms, Cardinal *nparms)
{
    AuthWidget aw = (AuthWidget) w;
    Prompt *prompt=aw->auth.prompts;
    int num;
    int prevnum;
    Prompt *prevprompt = NULL;
    XFontStruct *font;
    char *str;

    D(("called."));
    DrawCursor(w, aw->auth.cursorpos+2, TRUE);

    for (num=0, prompt=aw->auth.prompts;
	 num < aw->auth.where && prompt;
	 num++, prompt=(Prompt *)prompt->next )
	if ( prompt->type == ECHO || prompt->type == NOECHO ) {
	    prevnum = num;
	    prevprompt = prompt;
	}

    if ( prevprompt != NULL ) {
	prompt = prevprompt;
	aw->auth.where = prevnum;
    }

    font = aw->auth.entry_font;
    str = prompt->entrytext;
    if ( prompt->type != NOECHO )
	aw->auth.cursorpos = XTextWidth(font, str, strlen(str));
    else
	aw->auth.cursorpos = 0;

    DrawCursor(w, aw->auth.cursorpos+2, FALSE);

    D(("end."));
}
