/* Copyright (C) 2002, 2003, 2004 Thorsten Kukuk
   Author: Thorsten Kukuk <kukuk@suse.de>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License version 2 as
   published by the Free Software Foundation.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

#include <ctype.h>
#include <netdb.h>

#ifdef USE_SLP
#include <slp.h>
#endif


static int
start_request (SSL *ssl, char *username, int admin_mode)
{
  request_header req;
  char *locale = getenv ("LANG");

  if (admin_mode)
    req.request = START_ADMIN;
  else
    req.request = START;
  req.version = RPASSWD_VERSION;
  req.data_len = strlen (username) + 1;
  if (locale)
    req.locale_len = strlen (locale) + 1;
  else
    req.locale_len = 0;

  if (SSL_write (ssl, &req, sizeof (request_header)) !=
      sizeof (request_header))
    return -1;

  if (locale)
    if (SSL_write (ssl, locale, req.locale_len) != req.locale_len)
      return -1;

  if (SSL_write (ssl, username, req.data_len) != req.data_len)
    return -1;

  return 0;
}

static int
send_string (SSL *ssl, u_int32_t ret, const char *str)
{
  conv_header resp;

  resp.retval = ret;
  if (str == NULL)
    resp.data_len = 0;
  else
    resp.data_len = strlen (str) + 1;
  if (TEMP_FAILURE_RETRY (SSL_write (ssl, &resp, sizeof (resp)))
      != sizeof (resp))
    return E_FAILURE;

  if (str)
    if (TEMP_FAILURE_RETRY (SSL_write (ssl, str, resp.data_len))
	!= resp.data_len)
      return E_FAILURE;

  return E_SUCCESS;
}

#ifdef USE_SLP
/* Ask SLP server for rpasswd service.  */

struct slpcb {
  char *srvurl;
  SLPError err;
};

static SLPBoolean
MySLPSrvURLCallback (SLPHandle hslp __attribute__((unused)),
		     const char *srvurl,
		     unsigned short lifetime __attribute__((unused)),
		     SLPError errcode, void *cookie)
{
  struct slpcb *cb = (struct slpcb *) cookie;

  if (errcode == SLP_OK)
    {
      cb->srvurl = strdup (srvurl);
      cb->err = SLP_OK;
      /* We don't won't more currently.  */
      return SLP_FALSE;
    }
  else
    cb->err = errcode;

  return SLP_FALSE; /* We don't wan't to be called again.  */
}

static int
query_slp (HANDLE char **hostp, char **portp)
{
  struct slpcb callbackres;
  SLPError err;
  SLPHandle hslp;


  *hostp = NULL;
  *portp = NULL;

  err = SLPOpen ("en", SLP_FALSE, &hslp);
  if (err != SLP_OK)
    {
      PRINTF (ERR_HANDLE, _("Error opening SLP handle: %i.\n"), err);
      return err;
    }

  err = SLPFindSrvs (hslp, "rpasswd.thkukuk", 0, 0,
		     MySLPSrvURLCallback, &callbackres);

  /* err may contain an error code that occurred as the slp library
     _prepared_ to make the call.  */
  if (err != SLP_OK || callbackres.err != SLP_OK)
    {
      PRINTF (STD_HANDLE, _("No service found with SLP.\n"));
      return -1;
    }

  /* Now that we're done using slp, close the slp handle */
  SLPClose (hslp);

  if (callbackres.srvurl != NULL)
    {
      *hostp = callbackres.srvurl + 26;
      *portp = strchr (*hostp, ':');
      if (*portp)
	{
	  char *cp;

	  *portp[0] = '\0';
	  *portp += 1;
	  cp = *portp;
	  while (isdigit (*cp))
	    {
	      cp++;
	    }
	  if (*cp != '\0')
	    *cp = '\0';
	}
      return 0;
    }
  return -1;
}
#endif

static int
parse_reqcert (const char *str)
{
  if (strcmp (str, "never") == 0)
    return 0;
  else if (strcmp (str, "allow") == 0)
    return 1;
  else if (strcmp (str, "try") == 0)
    return 2;
  else if (strcmp (str, "demand") == 0 || strcmp (str, "hard") == 0)
    return 3;

  /* If we cannot parse it, use saftest mode.  */
  return 3;
}

/* Load the config file (/etc/rpasswd.conf)  */
static int
load_config (const char *configfile, char **hostp, char **portp,
	     int *reqcertp
#ifdef DO_VERBOSE_OUTPUT
	     ,int verbose, int check_syntax
#endif
	     )
{
  FILE *fp;
  char *buf = NULL;
  size_t buflen = 0;
  int have_entries = 0;		/* # of entries we found in config file */
#ifdef DO_VERBOSE_OUTPUT
  int bad_entries = 0;
#endif

  fp = fopen (configfile, "r");
  if (NULL == fp)
    return 1;

#ifdef DO_VERBOSE_OUTPUT
  if (verbose > 1)
    PRINTF (STD_HANDLE, _("parsing config file"));
#endif

  while (!feof (fp))
    {
      char *tmp, *cp;
#if defined(HAVE_GETLINE)
      ssize_t n = getline (&buf, &buflen, fp);
#elif defined (HAVE_GETDELIM)
      ssize_t n = getdelim (&buf, &buflen, '\n', fp);
#else
      ssize_t n;

      if (buf == NULL)
	{
	  buflen = 8096;
	  buf = malloc (buflen);
	}
      buf[0] = '\0';
      fgets (buf, buflen - 1, fp);
      if (buf != NULL)
	n = strlen (buf);
      else
	n = 0;
#endif /* HAVE_GETLINE / HAVE_GETDELIM */
      cp = buf;

      if (n < 1)
	break;

      tmp = strchr (cp, '#');	/* remove comments */
      if (tmp)
	*tmp = '\0';
      while (isspace ((int) *cp))	/* remove spaces and tabs */
	++cp;
      if (*cp == '\0')		/* ignore empty lines */
	continue;

      if (cp[strlen (cp) - 1] == '\n')
	cp[strlen (cp) - 1] = '\0';

#ifdef DO_VERBOSE_OUTPUT
      if (verbose > 1)
	PRINTF (STD_HANDLE, "%s %s", _("Trying entry:"), cp);

      if (check_syntax)
	PRINTF (STD_HANDLE, "%s %s\n", _("Trying entry:"), cp);
#endif

      if (strncmp (cp, "server", 6) == 0 && isspace ((int) cp[6]))
	{
	  if (hostp != NULL)
	    {
	      char tmpserver[MAXHOSTNAMELEN + 1];

	      if (sscanf (cp, "server %s", tmpserver) == 1)
		*hostp = strdup (tmpserver);
	    }
	  continue;
	}
      else if (strncmp (cp, "port", 4) == 0 && isspace ((int) cp[4]))
	{
	  if (portp != NULL)
	    {
	      char tmpport[30];

	      if (sscanf (cp, "port %s", tmpport) == 1)
		*portp = strdup (tmpport);
	    }
	  continue;
	}
      else if (strncmp (cp, "reqcert", 7) == 0 && isspace ((int) cp[7]))
	{
	  char *p = &cp[7];

	  while (isspace (*p))
	    ++p;

	  *reqcertp = parse_reqcert (p);
	  continue;
	}

#ifdef DO_VERBOSE_OUTPUT
      if (check_syntax)
	{
	  PRINTF (STD_HANDLE, _("Entry \"%s\" is not valid!\n"), cp);
	  ++bad_entries;
	}
      else
        PRINTF (ERR_HANDLE, _("Entry \"%s\" is not valid, ignored!\n"), cp);
#endif
    }
  fclose (fp);

  if (buf)
    free (buf);

#ifdef DO_VERBOSE_OUTPUT
  if (check_syntax)
    {
      if (bad_entries)
	{
	  PRINTF (STD_HANDLE, _("Bad entries found.\n"));
	  return 1;
	}
      if (!have_entries)
	{
	  PRINTF (STD_HANDLE, _("No entry found.\n"));
	  return 1;
	}
    }
#endif

  if (!have_entries)
    {
#ifdef DO_VERBOSE_OUTPUT
      if (verbose > 1)
	PRINTF (STD_HANDLE, _("No entry found."));
#endif
      return 1;
    }

  return 0;
}

static int
connect_to_server (HANDLE const char *hostp, const char *portp,
		   int family)
{
#ifdef NI_WITHSCOPEID
  const int niflags = NI_NUMERICHOST | NI_WITHSCOPEID;
#else
  const int niflags = NI_NUMERICHOST;
#endif
  struct addrinfo hints, *res, *res0;
  int error;
  int sock = -1;

  memset (&hints, 0, sizeof (hints));
  hints.ai_family = family;

  hints.ai_socktype = SOCK_STREAM;
  hints.ai_flags = AI_CANONNAME;

  error = getaddrinfo (hostp, portp, &hints, &res0);
  if (error)
    {
      if (error == EAI_NONAME)
	{
	  PRINTF (ERR_HANDLE,
		  _("Hostname or service not known for specified protocol\n"));
	  return -1;
	}
      else if (error == EAI_SERVICE)
	{
	  /* if port cannot be resolved, try compiled in
	     port number. If this works, don't abort here.  */
	  char *cp;
	  asprintf (&cp, "%d", RPASSWDD_PORT);
	  error = getaddrinfo (hostp, cp, &hints, &res0);
	  free (cp);
	  if (error)
	    {
	      PRINTF (ERR_HANDLE, _("bad port: %s\n"), portp);
	      return -1;
	    }
	}
      else
	{
	  PRINTF (ERR_HANDLE, "%s: %s\n", hostp, gai_strerror (error));
	  return -1;
	}
    }

  for (res = res0; res; res = res->ai_next)
    {
      char hbuf[NI_MAXHOST];

      if (getnameinfo (res->ai_addr, res->ai_addrlen,
		       hbuf, sizeof (hbuf), NULL, 0, niflags) != 0)
	strcpy (hbuf, "(invalid)");
      switch (res->ai_family)
	{
	case AF_INET:
	  {
	    struct sockaddr_in s_in;
	    memcpy (&s_in, res->ai_addr, sizeof (struct sockaddr_in));
	    PRINTF (STD_HANDLE, _("Trying %s port %d...\r\n"),
		    hbuf, ntohs (s_in.sin_port));
	    break;
	  }
	case AF_INET6:
	  {
	    struct sockaddr_in6 s_in6;
	    memcpy (&s_in6, res->ai_addr, sizeof (struct sockaddr_in));
	    PRINTF (STD_HANDLE, _("Trying %s port %d...\r\n"),
		    hbuf, ntohs (s_in6.sin6_port));
	    break;
	  }
	default:
	  PRINTF (STD_HANDLE, _("Trying %s...\r\n"), hbuf);
	  break;
	}

      /* Create the socket.  */
      sock = socket (res->ai_family, res->ai_socktype, res->ai_protocol);
      if (sock < 0)
	continue;

      if (connect (sock, res->ai_addr, res->ai_addrlen) < 0)
	{
	  if (getnameinfo (res->ai_addr, res->ai_addrlen,
			   hbuf, sizeof (hbuf), NULL, 0, niflags) != 0)
	    strcpy (hbuf, "(invalid)");
	  PRINTF (ERR_HANDLE, _("connect to address %s: %s\n"), hbuf,
		   strerror (errno));
	  close (sock);
	  sock = -1;
	  continue;
	}
      PRINTF (STD_HANDLE, "\n");
      break;
    }
  freeaddrinfo (res0);
  return sock;
}

static int
start_ssl (HANDLE int sock, int reqcert, int verbose,
	   SSL_CTX **ctx, SSL **ssl)
{
  X509 *server_cert;
  char *str;
  SSL_METHOD *meth;
  long verify_result;
  int err;

  SSLeay_add_ssl_algorithms ();
  meth = SSLv23_client_method ();
  SSL_load_error_strings ();
  *ctx = SSL_CTX_new (meth);
  if (*ctx == NULL)
    {
      PRINTF (ERR_HANDLE, ERR_error_string (ERR_get_error (), NULL));
      return E_SSL_FAILURE;
    }

#if 0
  /* This is only necessary if we configure a unusual path.
     XXX Make this a program option.  */
  if (!SSL_CTX_load_verify_locations (*ctx, NULL, "/etc/ssl/certs"))
    {
      PRINTF (ERR_HANDLE, _("error loading default verify locations: %s\n"),
	       ERR_error_string (ERR_get_error (), NULL));
      if (reqcert > 1)
	return E_SSL_FAILURE;
    }
#endif
  if (!SSL_CTX_set_default_verify_paths (*ctx))
    {
      PRINTF (ERR_HANDLE, _("error setting default verify path: %s\n"),
	       ERR_error_string (ERR_get_error (), NULL));
      if (reqcert > 1)
	return E_SSL_FAILURE;
    }

  /* Now we have TCP conncetion. Start SSL negotiation. */
  *ssl = SSL_new (*ctx);
  if (*ssl == NULL)
    {
      PRINTF (ERR_HANDLE, ERR_error_string (ERR_get_error (), NULL));
      return E_SSL_FAILURE;
    }
  SSL_set_fd (*ssl, sock);
  err = SSL_connect (*ssl);
  if (err < 1)
    {
      PRINTF (ERR_HANDLE, "SSL_connect: %s", ERR_error_string (err, NULL));
      close (sock);
      return E_SSL_FAILURE;
    }

  if (reqcert > 0 || verbose)
    {
      /* Get server's certificate (note: beware of dynamic allocation).  */
      server_cert = SSL_get_peer_certificate (*ssl);

      /* Verify severs certificate.  */
      verify_result = SSL_get_verify_result (*ssl);

      /* Following two steps are optional and not required for
	 data exchange to be successful except the client couldn't verfiy
	 the server certificate.  */
      if (verify_result || verbose)
	{
	  /* Get the cipher.  */
	  PRINTF (STD_HANDLE, _("SSL connection using %s\n\n"),
		  SSL_get_cipher (*ssl));

	  if (server_cert == NULL)
	    {
	      PRINTF (ERR_HANDLE, _("Server does not have a certificate?\n"));
	      if (reqcert >= 3)
		return E_SSL_FAILURE;
	    }
	  else
	    {
	      PRINTF (STD_HANDLE, _("Server certificate:\n"));

	      str = X509_NAME_oneline (X509_get_subject_name (server_cert),
				       0, 0);
	      if (str)
		{
		  PRINTF (STD_HANDLE, _("  subject: %s\n"), str);
		  free (str);
		}
	      str = X509_NAME_oneline (X509_get_issuer_name (server_cert),
				       0, 0);
	      if (str)
		{
		  PRINTF (STD_HANDLE, _("  issuer: %s\n"), str);
		  free (str);
		}
	      /* We could do all sorts of certificate verification stuff
		 here before deallocating the certificate.  */

	      PRINTF (STD_HANDLE, "\n");
	    }
	}

      if ((verify_result = SSL_get_verify_result (*ssl)) != X509_V_OK)
	{
	  PRINTF (ERR_HANDLE, "Server certificate is not ok: %s!\n",
		   X509_verify_cert_error_string (verify_result));
	  if (reqcert >= 2)
	    return E_SSL_FAILURE;
	}

      X509_free (server_cert);
    }

  return 0;
}
