/* $Id: flock.c,v 1.4 2003/03/17 15:17:48 hpa Exp $ */
/* ----------------------------------------------------------------------- *
 *   
 *   Copyright 2003 H. Peter Anvin - All Rights Reserved
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, Inc., 53 Temple Place Ste 330,
 *   Bostom MA 02111-1307, USA; either version 2 of the License, or
 *   (at your option) any later version; incorporated herein by reference.
 *
 * ----------------------------------------------------------------------- */

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <getopt.h>
#include <sysexits.h>
#include <sys/file.h>

const struct option long_options[] = {
  { "shared",       0, NULL, 's' },
  { "exclusive",    0, NULL, 'x' },
  { "unlock",       0, NULL, 'u' },
  { "nonblocking",  0, NULL, 'n' },
  { "nb",           0, NULL, 'n' },
  { "help",         0, NULL, 'h' },
  { "version",      0, NULL, 'V' },
  { 0, 0, 0, 0 }
};

const char *program;

void usage(int ex)
{
  fprintf(stderr,
	  "flock version " VERSION "\n"
	  "Usage: %s [options] fd#\n",
	  "  -s  --shared     Get a shared lock\n"
	  "  -e  --exclusive  Get an exclusive lock\n"
	  "  -u  --unlock     Remove a lock\n"
	  "  -n  --nonblock   Fail rather than wait\n"
	  "  -h  --help       Display this text\n"
	  "  -V  --version    Display version\n",
	  program);
  exit(ex);
}


int main(int argc, char *argv[])
{
  int type = LOCK_EX;
  int block = 0;
  int fd;
  int opt, ix;
  int err;
  char *eon;

  program = argv[0];

  if ( argc < 2 )
    usage(EX_USAGE);
  
  while ( (opt = getopt_long(argc, argv, "sexnuh", long_options, &ix)) != EOF ) {
    switch(opt) {
    case 's':
      type = LOCK_SH;
      break;
    case 'e':
    case 'x':
      type = LOCK_EX;
      break;
    case 'u':
      type = LOCK_UN;
      break;
    case 'n':
      block = LOCK_NB;
      break;
    case 'V':
      fprintf(stderr, "flock " VERSION "\n");
      exit(0);
    default:
      usage(0);
      break;
    }
  }

  fd = (int)strtol(argv[optind], &eon, 10);
  if ( *eon || !argv[optind] ) {
    fprintf(stderr, "%s: bad number: %s\n", program, argv[optind]);
    exit(EX_USAGE);
  }
  
  while ( flock(fd, type|block) ) {
    switch( (err = errno) ) {
    case EWOULDBLOCK:		/* -n option set and failed to lock */
      exit(1);
    case EINTR:			/* Interrupted, try again */
      continue;
    default:			/* Other errors */
      fprintf(stderr, "%s: %d: %s\n", program, fd, strerror(err));
      exit((err == ENOLCK||err == ENOMEM) ? EX_OSERR : EX_DATAERR);
    }
  }
  
  return 0;			/* If we get here, we were successful */
}
